<?php

namespace App\Http\Controllers;
use DB;
use App\Quotation;
use Illuminate\Http\Request;
use App\Patients;
use App\Doctors;
use Session;
use App\Questionnaire;
use Illuminate\Support\Facades\Auth;
use Redirect;
use Carbon\Carbon;
use Carbon\CarbonPeriod;

use PDF;
use Illuminate\Support\Facades\Hash;
use ZipArchive;
class HrController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Http\Response
     */
    
    public function showHolidays(Request $request)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        
        $year = $request->input('year', Carbon::now()->year);
        $month = $request->input('month', Carbon::now()->month);

        // Determine if we need to go to previous or next month
        if ($request->input('action') === 'prev') {
            $firstDayOfMonth = Carbon::create($year, $month)->subMonth()->startOfMonth();
        } elseif ($request->input('action') === 'next') {
            $firstDayOfMonth = Carbon::create($year, $month)->addMonth()->startOfMonth();
        } else {
            $firstDayOfMonth = Carbon::create($year, $month, 1);
        }

        $lastDayOfMonth = $firstDayOfMonth->copy()->endOfMonth();
        $startOfCalendar = $firstDayOfMonth->copy()->startOfMonth()->startOfWeek();
        $endOfCalendar = $lastDayOfMonth->copy()->endOfMonth()->endOfWeek();

        $calendar = [];
        $currentDay = $startOfCalendar->copy();

        while ($currentDay <= $endOfCalendar) {
            $week = [];
            for ($i = 0; $i < 7; $i++) {
                $week[] = $currentDay->copy();
                $currentDay->addDay();
            }
            $calendar[] = $week;
        }
        
        

        return view('hr.holidays', [
            'calendar' => $calendar,
            'year' => $firstDayOfMonth->year,
            'month' => $firstDayOfMonth->month,
            'firstDayOfMonth' => $firstDayOfMonth,
            'lastDayOfMonth' => $lastDayOfMonth,
            'monthName' => $firstDayOfMonth->format('F, Y'),
        ]);
    }
    
    
    public static function getAbsents($dateToCheck)
    {
        $holidays =  DB::table('holidays as a')
            ->leftjoin('users as b', 'a.user_id', '=', 'b.id')
            ->select('b.firstname', 'b.lastname', 'b.image', 'a.*')
            ->where(function($query) use ($dateToCheck) {
                $query->whereDate('a.from_date', '<=', $dateToCheck)
                      ->whereDate('a.to_date', '>=', $dateToCheck);
            })
        ->get();
        
        return $holidays;
    }
    
    public function showDashboard(Request $request)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        
        $handbook = DB::table('user_handbooks')->orderBy('id', 'desc')
            ->limit(1)
            ->first();
        $users = DB::table('users')->orderBy('firstname', 'asc')
            ->where('role', '!=', 'admin')
            ->get();
        
        return view('hr.dashboard', ['users'=>$users, 'handbook'=>$handbook]);
    }
    
    public function showContract(Request $request, $id)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        
      
        $user = DB::table('users')->where('id', $id)->first();
        
        return view('hr.contract', ['user'=>$user]);
    }
    
    
    public function showUserDetails(Request $request, $id)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        $user = DB::table('users')->where('id', $id)->first();
        return view('hr.personal_info', ['user' => $user]);
    }
    
    public function showDocuments(Request $request, $id)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        $user = DB::table('users')->where('id', $id)->first();
        $staffs = DB::table('users')->where('role', '!=', 'admin')->get();
        $documents = DB::table('user_documents')->where('user_id', $id)->get();
        foreach($documents as $document){
            $document->staff = DB::table('user_document_staff as a')
                ->leftjoin('users as b', 'a.staff_id', '=', 'b.id')
                 ->select('b.firstname', 'b.lastname', 'b.email', 'b.image')
                ->where('a.document_id', $document->id)
                ->get();
        }
        
        return view('hr.documents', ['user' => $user, 'documents'=>$documents, 'staffs'=>$staffs]);
    }
    
    public function showAttendance(Request $request, $id)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        $user = DB::table('users')->where('id', $id)->first();
        
        $holidays = DB::table('holidays')->where('user_id', $id)->get();
        $totalHolidays = 0;
        $totalLeaves = 0;
        foreach($holidays as $holiday){
            if($holiday->status=='accepted'){
                $fromDate = Carbon::parse($holiday->from_date);
                $toDate = Carbon::parse($holiday->to_date);
                if($fromDate==$toDate){
                    $days = 0.5;
                }else{
                    $period = CarbonPeriod::create($fromDate, $toDate);
                    $days = $period->filter(function ($date) {
                        return !$date->isWeekend(); // Exclude Saturdays and Sundays
                    })->count();
                }
                $totalHolidays+=$days;
            }
        }

        
        
        $sickleaves = DB::table('sick_leaves')->where('user_id', $id)->get();
        foreach($sickleaves as $leave){
            if($fromDate==$toDate){
                $days = 0.5;
            }else{
                $fromDate = Carbon::parse($leave->from_date);
                $toDate = Carbon::parse($leave->to_date);
                $days = $fromDate->diffInDays($toDate);
            }

            
            $totalLeaves+=$days;
        }
        return view('hr.attendance', ['user' => $user, 'holidays'=>$holidays, 'sick_leaves'=>$sickleaves, 'totalHolidays'=>$totalHolidays, 'totalLeaves'=>$totalLeaves]);
    }
    
    
    
    
    public function showProfessionalDetails(Request $request, $id)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        $user = DB::table('users')->where('id', $id)->first();
        return view('hr.professional_info', ['user' => $user]);
    }
    
    public function showPPE(Request $request, $id)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        $user = DB::table('users')->where('id', $id)->first();
        $workwears = DB::table('workwears')->where('user_id', $id)->get();
        $ppes = DB::table('ppes')->where('user_id', $id)->get();
        return view('hr.ppe_register', ['user' => $user, 'workwears'=>$workwears, 'ppes'=>$ppes]);
    }
    
    public function showAssetRegister(Request $request, $id)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        $user = DB::table('users')->where('id', $id)->first();
        $assets = DB::table('user_assets')->where('user_id', $id)->get();
        return view('hr.asset_register', ['user' => $user, 'assets'=>$assets]);
    }
    
    public function deleteAsset(Request $request)
    {
        DB::table('user_assets')->where('id', $request->delete_id)->delete();
        
        $request->session()->flash('success', 'Asset removed successfully.');
        return redirect()->back();
    }
    
    public function editUser(Request $request, $id)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        $user = DB::table('users')->where('id', $id)->first();
        return view('hr.edit_user', ['user' => $user]);
    }
    
    public function editProfessionalInfo(Request $request, $id)
    {
        if(Auth::user()->role!='admin'){
            abort(403);
        }
        $project_types = DB::table('project_types')->get();
        $user = DB::table('users')->where('id', $id)->first();
        return view('hr.edit_professional_info', ['user' => $user, 'project_types'=>$project_types]);
    }
    
    public function enableNotify(Request $request, $id)
    {
        DB::table('user_documents')->where('id', $request->document_id)->update(
             array(
                    'notify' => $request->switch_status,
             )
        );
        
        echo 1;
    }
    
    public function enableAssetNotify(Request $request, $id)
    {
        DB::table('user_assets')->where('id', $request->document_id)->update(
             array(
                    'notify' => $request->switch_status,
             )
        );
        
        echo 1;
    }
    
    
    public function uploadContract(Request $request, $id)
    {
        $request->validate([
            'document' => 'required|mimes:pdf|max:3072',
        ]);

        if ($request->hasFile('document')) {
            $file = $request->file('document');
            $ext = $file->getClientOriginalExtension();
            $filename = uniqid(time()) . '.' . $ext;
            $name = $file->getClientOriginalName();
            $file->move(public_path('documents'), $filename);
            $fileid = DB::table('users')->where('id', $id)->update(
                 array(
                        'contract'   =>   $filename,
                 )
            );

            
            
            $request->session()->flash('success', 'Contract uploaded successfully.');
            return redirect()->back();
        }
    }
    
    public function uploadHandbook(Request $request)
    {
        $request->validate([
            'document' => 'required|mimes:pdf|max:2048',
        ]);

        if ($request->hasFile('document')) {
            $file = $request->file('document');
            $ext = $file->getClientOriginalExtension();
            $filename = uniqid(time()) . '.' . $ext;
            $name = $file->getClientOriginalName();
            $file->move(public_path('documents'), $filename);
            $added = date('Y-m-d H:i');
            $fileid = DB::table('user_handbooks')->insertGetId(
                 array(
                        'name'   =>   $name,
                        'file'   =>   $filename,
                       'created_at'   =>   $added,
                        
                 )
            );

            
            
            $request->session()->flash('success', 'PDF uploaded successfully.');
            return redirect()->back();
        }
    }
    
    public function addHoliday(Request $request, $id)
    {
        
        // Validation rules
        if($request->type == 'half'){
            $rules = [
                'date' => 'required|date',
                'time' => 'required'
            ];
        } else {
            $rules = [
                'from' => 'required|date',
                'to' => 'required|date|after_or_equal:from',
            ];
        }

        // Validate the input
        $validated = $request->validate($rules);
        
        if($request->type=='half'){
            $from = $request->date;
            $to = $request->date;
            $time = $request->time;
        }else{
            $from = $request->from;
            $to = $request->to;
            $time = "";
        }
        
        $fileid = DB::table('holidays')->insertGetId(
             array(
                    'user_id'   =>   $id,
                    'from_date'   =>   $from,
                    'to_date'   =>   $to,
                    'type' => $request->type,
                    'description'   =>   $request->description,
                    'time'   =>   $time,
             )
        );

        $request->session()->flash('success', 'Holiday added successfully.');
        return redirect()->back();

    }
    
    
    public function addPPE (Request $request, $id)
    {
        
        DB::table('ppes')->insertGetId(
             array(
                    'user_id'   =>   $id,
                    'name'   =>   $request->name,
             )
        );

        $request->session()->flash('success', 'PPE added successfully.');
        return redirect()->back();

    }
    
    
    public function addWorkwear (Request $request, $id)
    {
        
        DB::table('workwears')->insertGetId(
             array(
                    'user_id'   =>   $id,
                    'name'   =>   $request->name,
             )
        );

        $request->session()->flash('success', 'Workwear added successfully.');
        return redirect()->back();

    }
    
    
    public function updateWorkwear(Request $request, $id)
    {
        
        DB::table('workwears')
            ->where('id', $request->workwear_id)
            ->where('user_id', $id)
            ->update(
             array(
                    'received'   =>   1,
                    'received_date'   =>   date('Y-m-d H:i'),
             )
        );

        $request->session()->flash('success', 'Received successfully.');
        return redirect()->back();

    }
    
    public function updatePPE(Request $request, $id)
    {
        
        DB::table('ppes')
            ->where('id', $request->ppe_id)
            ->where('user_id', $id)
            ->update(
             array(
                    'received'   =>   1,
                    'received_date'   =>   date('Y-m-d H:i'),
             )
        );

        $request->session()->flash('success', 'Received successfully.');
        return redirect()->back();

    }
    
    public function changeHolidayStatus(Request $request, $id)
    {
        
        $fileid = DB::table('holidays')
            ->where('id', $request->holiday_id)
            ->where('user_id', $id)
            ->update(
             array(
                    'status'   =>   $request->status,
             )
        );

        $request->session()->flash('success', 'Updated successfully.');
        return redirect()->back();

    }
    
    public function addSickLeave(Request $request, $id)
    {
        $request->validate([
            'document' => 'required|mimes:jpeg,png,jpg,gif,pdf,doc,docx,xls,xlsx,ppt,pptx,txt,zip|max:2048',
        ]);

        if ($request->hasFile('document')) {
            $file = $request->file('document');
            $ext = $file->getClientOriginalExtension();
            $filename = uniqid(time()) . '.' . $ext;
            $name = $file->getClientOriginalName();
            $file->move(public_path('documents'), $filename);
            $added = date('Y-m-d H:i');
            
            if($request->type=='half'){
                $from = $request->date;
                $to = $request->date;
            }else{
                $from = $request->from;
                $to = $request->to;
            }
            
            $fileid = DB::table('sick_leaves')->insertGetId(
                 array(
                        'user_id'   =>   $id,
                        'description'   =>   $request->description,
                        'from_date'   =>   $from,
                        'to_date'   =>   $to,
                        'type'   =>   $request->type,
                        'document_file'   =>   $filename,
                        'document_name'   =>   $name,                        
                 )
            );

            $request->session()->flash('success', 'Sick leave added successfully.');
            return redirect()->back();
        }
    }
    
    public function addAsset(Request $request, $id)
    {
        $request->validate([
            'document' => 'required|mimes:jpeg,png,jpg,gif,pdf,doc,docx,xls,xlsx,ppt,pptx,txt,zip|max:2048',
        ]);

        if ($request->hasFile('document')) {
            $file = $request->file('document');
            $ext = $file->getClientOriginalExtension();
            $filename = uniqid(time()) . '.' . $ext;
            $name = $file->getClientOriginalName();
            $file->move(public_path('documents'), $filename);
            $added = date('Y-m-d H:i');
            
            if($request->reminder_type=='weekly'){
                $reminderDay = $request->weekly_reminder;
            }elseif($request->reminder_type=='monthly'){
                $reminderDay = $request->monthly_reminder;
            }elseif($request->reminder_type=='yearly'){
                $reminderDay = $request->yearly_reminder;
            }
            
            $fileid = DB::table('user_assets')->insertGetId(
                 array(
                        'user_id'   =>   $id,
                        'item_name'   =>   $request->item_name,
                        'document_file'   =>   $filename,
                        'document_name'   =>   $name,
                        'date'   =>   $request->date,
                        'documentation'   =>   $request->documentation,
                        'notes'   =>   $request->notes,
                        'reminder_type'   =>   $request->reminder_type,
                        'reminder_day'   =>   $reminderDay,
                        
                 )
            );

            
            
            $request->session()->flash('success', 'Asset added successfully.');
            return redirect()->back();
        }
    }
    
    public function addDocument(Request $request, $id)
    {
        $request->validate([
            'document' => 'required|mimes:jpeg,png,jpg,gif,pdf,doc,docx,xls,xlsx,ppt,pptx,txt,zip|max:2048',
        ]);

        if ($request->hasFile('document')) {
            $file = $request->file('document');
            $ext = $file->getClientOriginalExtension();
            $filename = uniqid(time()) . '.' . $ext;
            $name = $file->getClientOriginalName();
            $file->move(public_path('documents'), $filename);
            $added = date('Y-m-d H:i');
            $fileid = DB::table('user_documents')->insertGetId(
                 array(
                        'user_id'   =>   $id,
                        'name'   =>   $request->title,
                        'file'   =>   $filename,
                        'filename'   =>   $name,
                        'created_at'   =>   $added,
                        'expiry_date'   =>   $request->expiry_date,
                        
                 )
            );

            
            
            $request->session()->flash('success', 'Document added successfully.');
            return redirect()->back();
        }
    }
    
    
    public function updateProfessionalInfo(Request $request, $id)
    {
        if(isset($request->project_types)){
            $project_types = implode(',', $request->project_types);
        }else{
            $project_types = 0;
        }
        DB::table('users')->where('id', $id)->update(
             array(
                    'project_types_access' => $project_types,
                    'employee_id'   =>   $request->employee_id,
                    'hourly_rate'   =>   $request->hourly_rate,
                    'uplift_amount'   =>   $request->uplift_amount,
                    'work_email'   =>   $request->work_email,
                    'office_location'   =>   $request->office_location,
                    'employee_type'   =>   $request->employee_type,
                    'department'   =>   $request->department,
                    'working_days'   =>   $request->working_days,
                    'joining_date'   =>   $request->joining_date,
             )
        );
        
        $request->session()->flash('success', 'User updated successfully.');
        return redirect()->back();
    }
    
    
    public function updateUser(Request $request, $id)
    {
        DB::table('users')->where('id', $id)->update(
             array(
                    'firstname'   =>   $request->firstname,
                    'lastname'   =>   $request->lastname,
                    'email'   =>   $request->email,
                    'phone'   =>   $request->phone,
                    'gender'   =>   $request->gender,
                    'marital_status'   =>   $request->marital_status,
                    'dob'   =>   $request->dob,
                    'nationality' => $request->nationality,
                    'address' => $request->address,
                    'city' => $request->city,
                    'country' => $request->country,
                    'postcode' => $request->postcode,
             )
        );
        
        $request->session()->flash('success', 'User updated successfully.');
        return redirect()->back();
    }
    
    public function updateHolidays(Request $request, $id)
    {
        DB::table('users')->where('id', $id)->update(
             array(
                    
                    'total_holidays' => $request->total_holidays,
             )
        );
        
        $request->session()->flash('success', 'Holidays updated successfully.');
        return redirect()->back();
    }
    
    
    public function addDocumentStaff(Request $request, $id)
    {
        DB::table('user_document_staff')
            ->insert(
             array(
                    'user_id'   =>   $id,
                    'document_id'   =>   $request->document_id,
                    'staff_id'   =>   $request->staff_id,
             )
        );
        
        $request->session()->flash('success', 'Staff added successfully.');
        return redirect()->back();
    }

    

    
}